"use client";

import { useEffect, type ReactNode } from "react";
import { usePathname, useRouter } from "next/navigation";
import { useAuth } from "@/contexts/AuthContext";

interface ProtectedRouteProps {
    children: ReactNode;
    fallback?: ReactNode;
}

function buildRedirectPath(pathname: string | null, queryString: string): string {
    const safePath = pathname || "/";
    const query = queryString.startsWith("?") ? queryString.slice(1) : queryString;
    return query ? `${safePath}?${query}` : safePath;
}

export function ProtectedRoute({ children, fallback = null }: ProtectedRouteProps) {
    const { isAuthenticated, loading } = useAuth();
    const router = useRouter();
    const pathname = usePathname();
    const queryString = typeof window !== "undefined" ? window.location.search : "";
    const redirectPath = buildRedirectPath(pathname, queryString);

    useEffect(() => {
        if (loading || isAuthenticated) return;
        router.replace(`/auth?redirectTo=${encodeURIComponent(redirectPath)}`);
    }, [loading, isAuthenticated, redirectPath, router]);

    if (loading) return <>{fallback}</>;
    if (!isAuthenticated) return <>{fallback}</>;
    return <>{children}</>;
}
